<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    /**
     * Handle a login request to the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response|\Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        // 1. التحقق من أن المدخلات (الايميل وكلمة المرور) موجودة
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        // 2. محاولة تسجيل الدخول باستخدام البيانات
        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials)) {
            // 3. في حالة النجاح، قم بإنشاء Token للمستخدم
            /** @var \App\Models\User $user */
            $user = Auth::user();
            $token = $user->createToken('authToken')->plainTextToken;

            // 4. أرجع بيانات المستخدم مع الـ Token
            return response()->json([
                'user' => $user,
                'token' => $token,
            ], 200);
        } else {
            // 5. في حالة الفشل، أرجع رسالة خطأ
            return response()->json(['error' => 'Unauthorized'], 401);
        }
    }
    
    /**
     * دالة مؤقتة لتسجيل مستخدم جديد لاختبار API.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response|\Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()], 422);
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password), // هذا السطر هو الأهم: يقوم بتشفير كلمة المرور
        ]);

        $token = $user->createToken('authToken')->plainTextToken;

        return response()->json([
            'user' => $user,
            'token' => $token,
        ], 201);
    }
}